//*************************************************************************************************
//
//	Description:
//		ground.fx - Shader used for ground-cover materials. Supports three diffuse textures, two with
//								a seperate mapping channel, and a third scaled and rotated from the second, plus
//								specular and normal maps matching the most detailed layer.
//							  Vertex colours are multiplied into the base colour also.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     05/31/2006  0.1           Created
//		TNettleship     06/02/2006  0.2           Added layer 3 plus scaling
//		TNettleship     06/08/2006  0.3           Added seperate U/V scaling factors for layer 3.
//		TNettleship			06/27/2006	1.0						Added support for compressed normal maps. Changes
//																							to get it working in the engine.
//		TNettleship			10/03/2006	1.1						Fixed a bug with channel assignment & vertex colours
//																							in 3DSMAX.
//		TNettleship			10/16/2006	1.2						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.4						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	1.5						Fixed a bug with directional lighting.
//		TNettleship			11/21/2006	1.6						Added support for deferred rendering.
//		TMann						11/23/2006	1.11					Added _WIN_GL_ support
//		TMann						11/27/2006	1.12					Texture filtering for GL
//		TNettleship			12/04/2006	1.13					Added premultiplied alpha code to the fragment shader.
//		TMann						12/11/2006	1.14					PS3 version
//		TNettleship			01/02/2007	1.15					Stopped premultiplied alpha from being used in opaque
//																							materials.
//		TMann						01/09/2007	1.16					Added PS3/GL literal param passing
//		TNettleship			01/31/2007	1.17					Pixel shader optimisations
//		TNettleship			02/13/2007	1.18					Fixed normal map inconsistency between max & engine
//		BIrvine					02/13/2007	1.19					Temporary shadow implementation
//		TNettleship			04/18/2007	2.0						Added shader specialisation.
//		TNettleship			07/10/2007	2.01					Added support for vertex alpha.
//		TNettleship			07/11/2007	2.02					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.03          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.04					Removed texture bias, added anisotropy to mid/detail diffuse.
//		TNettleship			10/23/2007	2.05					Converted to on-load rendermode behaviour binding.
//		TNettleship			01/28/2008 2.06					Turned anisotropic filtering back on.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define PROPER_ADDITIVE
#define NORMAL_MAPPING
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture broadDiffuseTexture : DiffuseMap			// Broad diffuse colour in RGB
#else
texture broadDiffuseTexture : TEXTURE					// Broad diffuse colour in RGB
#endif
<
	string UIName = "Broad Diffuse";
	bool appEdit = true;
>;

texture middleDiffuseTexture : TEXTURE				// Middle diffuse colour in RGB
<
	string UIName = "Middle Diffuse";
	bool appEdit = true;
>;

texture detailDiffuseTexture : TEXTURE				// Detail diffuse colour in RGB
<
	string UIName = "Detail Diffuse";
	bool appEdit = true;
>;

float detailUScale														// Detail layer U scale
<
	string UIName = "Detail Layer U Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

float detailVScale														// Detail layer V scale
<
	string UIName = "Detail Layer V Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

bool additiveDetail														// Additive detail?
<
	string UIName = "Additive Detail?";
	bool appEdit = true;
> = true;

SPECIALISATION_PARAM( properAdditive, "Proper Additive?", "PROPER_ADDITIVE" )	// TRUE if true additive detail should be used

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Texture", useNormalMap )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D broadDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="broadDiffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < broadDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D middleDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="middleDiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < middleDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D detailDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="detailDiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < detailDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float3 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord - N.B. Max requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 1 texture coord
	float3 alpha		 : TEXCOORD3;													// Vertex alpha
#else
	float4 colour    : COLOR0;														// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord
	float2 texCoord1 : TEXCOORD1;													// UV channel 1 texture coord
#endif
	float3 normal    : NORMAL;														// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal  : BINORMAL;													// Object space normal
#endif
};


// Output structure (also input to the fragment shader)
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float4 texCoord12	: TEXCOORD1;												// UV coords for texture channels 1 and 2
	float3 normal			: TEXCOORD2;												// Normal vector (world space)
	float4 eye				: TEXCOORD3;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT GroundVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord0 = _input.texCoord0;
	_output.texCoord12.xy = _input.texCoord1;
	_output.texCoord12.zw = _input.texCoord1 * float2( detailUScale, detailVScale );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float4 texCoord12	: TEXCOORD1;												// UV coords for texture channels 1 and 2
	float3 normal			: TEXCOORD2;												// Normal vector (world space)
	float4 eye				: TEXCOORD3;												// Eye vector (world space)
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD4_centroid;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float4 texCoord12	: TEXCOORD1;												// UV coords for texture channels 1 and 2
	float3 normal			: TEXCOORD2_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD3_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
#ifdef _POST_PARSE_
PSOUTPUT GroundFragmentShader( PSINPUT _input )
#else
PSOUTPUT GroundFragmentShader( PSINPUT _input, uniform bool _premultiplyAlpha )
#endif
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float2 detailCoord = _input.texCoord12.zw;
	float4 broadDiffuseTexColour = tex2D( broadDiffuseMap, _input.texCoord0 );
	float4 middleDiffuseTexColour = tex2D( middleDiffuseMap, _input.texCoord12.xy );
	float4 detailDiffuseTexColour = tex2D( detailDiffuseMap, detailCoord );
	float4 specularTexColour = tex2D( specularMap, detailCoord );
	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;

	// Factor vertex alpha into the diffuse alpha
	broadDiffuseTexColour.a *= _input.colour.a;

	// Calculate base colour (combined from broad and detail diffuse)
	float4 diffuseTexColour;
	if ( additiveDetail )
	{
		DEPENDENT_CODE_START( properAdditive )
#if defined( _3DSMAX_ ) || defined( PROPER_ADDITIVE )
			diffuseTexColour = ( broadDiffuseTexColour * middleDiffuseTexColour ) + detailDiffuseTexColour;
#endif
		DEPENDENT_CODE_ELSE( properAdditive )
#if defined( _3DSMAX_ ) || !defined( PROPER_ADDITIVE )
			diffuseTexColour = ( ( broadDiffuseTexColour * middleDiffuseTexColour ) + detailDiffuseTexColour ) * 0.5f;
#endif
		DEPENDENT_CODE_END( properAdditive )
	}
	else
	{
		diffuseTexColour = broadDiffuseTexColour * middleDiffuseTexColour * detailDiffuseTexColour;
	}


	if ( _premultiplyAlpha )
	{
		// Premultiply the diffuse RGB by the diffuse alpha
		diffuseTexColour.rgb *= broadDiffuseTexColour.a;
	}

	float4 accumulatedColour = diffuseTexColour * _input.colour;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
	float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, detailCoord );
		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = broadDiffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}









//
// Low Detail Shaders
//



struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float2 texCoord2	: TEXCOORD2;												// UV coords for texture channel 2
};


VSOUTPUT_LD GroundLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord0 = _input.texCoord0;
	_output.texCoord1 = _input.texCoord1;

	// Calculate scaled version of texture channel 1
	_output.texCoord2 = _input.texCoord1 * float2( detailUScale, detailVScale );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );


	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
#ifdef _POST_PARSE_
PSOUTPUT GroundLowDetailFragmentShader( VSOUTPUT_LD _input )
#else
PSOUTPUT GroundLowDetailFragmentShader( VSOUTPUT_LD _input, uniform bool _premultiplyAlpha )
#endif
{
	PSOUTPUT _output;

	// Read textures
	float4 broadDiffuseTexColour = tex2D( broadDiffuseMap, _input.texCoord0 );
	float4 middleDiffuseTexColour = tex2D( middleDiffuseMap, _input.texCoord1 );
	float4 detailDiffuseTexColour = tex2D( detailDiffuseMap, _input.texCoord2 );

	// Factor vertex alpha into the diffuse alpha
	broadDiffuseTexColour.a *= _input.colour.a;

	// Calculate base colour (combined from broad and detail diffuse)
	float4 diffuseTexColour;
	if ( additiveDetail )
	{
		DEPENDENT_CODE_START( properAdditive )
#if defined( _3DSMAX_ ) || defined( PROPER_ADDITIVE )
			diffuseTexColour = ( broadDiffuseTexColour * middleDiffuseTexColour ) + detailDiffuseTexColour;
#endif
		DEPENDENT_CODE_ELSE( properAdditive )
#if defined( _3DSMAX_ ) || !defined( PROPER_ADDITIVE )
			diffuseTexColour = ( ( broadDiffuseTexColour * middleDiffuseTexColour ) + detailDiffuseTexColour ) * 0.5f;
#endif
		DEPENDENT_CODE_END( properAdditive )
	}
	else
	{
		diffuseTexColour = broadDiffuseTexColour * middleDiffuseTexColour * detailDiffuseTexColour;
	}

	if ( _premultiplyAlpha )
	{
		// Premultiply the diffuse RGB by the diffuse alpha
		diffuseTexColour.rgb *= broadDiffuseTexColour.a;
	}

	float4 accumulatedColour = diffuseTexColour * _input.colour;

	accumulatedColour.w = broadDiffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Ground
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Ground_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx GroundVertexShader();
		PixelShader = compile sce_fp_rsx GroundFragmentShader( false );
#else		
		VertexShader = compile vs_3_0 GroundVertexShader();
		PixelShader = compile ps_3_0 GroundFragmentShader( false );
#endif		
	}
}


technique Ground_Translucent
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_Translucent";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Ground_Translucent_LowDetail";
	int    lowDetailDeferredID = 2;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
		string ZFunc = "less";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx GroundVertexShader();
		PixelShader = compile sce_fp_rsx GroundFragmentShader( true );
#else		
		VertexShader = compile vs_3_0 GroundVertexShader();
		PixelShader = compile ps_3_0 GroundFragmentShader( true );
#endif		
	}
}



technique Ground_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx GroundLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx GroundLowDetailFragmentShader( false );
#else		
		VertexShader = compile vs_3_0 GroundLowDetailVertexShader();
		PixelShader = compile ps_3_0 GroundLowDetailFragmentShader( false );
#endif		
	}
}


technique Ground_Translucent_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_Translucent_LowDetail";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx GroundLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx GroundLowDetailFragmentShader( true );
#else		
		VertexShader = compile vs_3_0 GroundLowDetailVertexShader();
		PixelShader = compile ps_3_0 GroundLowDetailFragmentShader( true );
#endif		
	}
}
